import { getServerSession } from 'next-auth';
import { authOptions } from '@/app/api/auth/[...nextauth]/route';
import { redirect } from 'next/navigation';
import { getAnnonceById } from '@/lib/services/annonceService';
import { getUserById } from '@/lib/services/userService';
import Link from 'next/link';
import ImageCarousel from '@/lib/components/ImageCarrousel';

export default async function AnnonceDetail({ params }: { params: { id: string } }) {
    const session = await getServerSession(authOptions);
    if (!session) redirect('/');

    const annonce = await getAnnonceById(Number(params.id));
    if (!annonce) return <div className="text-center p-20">Annonce non trouvée.</div>;

    const agent = await getUserById(annonce.agentId);
    const formattedPrice = new Intl.NumberFormat('fr-FR', { style: 'currency', currency: 'EUR', minimumFractionDigits: 0 }).format(annonce.prix);
    const imageSrc = annonce.imagePrincipale
        ? `data:image/jpeg;base64,${Buffer.from(annonce.imagePrincipale).toString("base64")}`
        : "/default-image-annonce.jpg";
    return (
        <div className="bg-white min-h-screen">
            <div className="container mx-auto p-4 sm:p-6 lg:p-8">
                <header className="py-4 mb-4">
                    <Link href="/" className="inline-flex items-center text-sm font-medium text-gray-500 hover:text-black mb-8">
                        &larr; Retour aux annonces
                    </Link>
                </header>

                <main className="grid grid-cols-1 lg:grid-cols-3 lg:gap-12">
                    {/* Colonne principale (Image et Description) */}
                    <div className="lg:col-span-2">

                        {/* Carrousel */}
                        <div className="mb-8">
                            <img src={imageSrc} alt={annonce.titre} className="w-full h-full object-cover"/>
                        </div>

                        {/* Titre et Adresse */}
                        <div className="border-b border-gray-200 pb-6 mb-6">
                            <h1 className="text-4xl font-bold text-gray-900">{annonce.titre}</h1>
                            <p className="text-md text-gray-500 mt-2">{annonce.address}, {annonce.ville}</p>
                        </div>

                        {/* Description */}
                        <h2 className="text-2xl font-semibold text-gray-800 mb-4">Description</h2>
                        <p className="text-gray-600 leading-relaxed">{annonce.description || 'Aucune description.'}</p>
                    </div>

                    {/* Prix et Agent */}
                    <aside className="mt-8 lg:mt-0">
                        <div className="sticky top-8 bg-white p-6 rounded-xl border border-gray-200 shadow-sm">
                            <p className="text-3xl font-bold text-gray-900 mb-6">{formattedPrice}</p>

                            <h3 className="text-xl font-semibold text-gray-800 border-t border-gray-200 pt-6">Agent en charge</h3>
                            {agent ? (
                                <div className="mt-4 space-y-3">
                                    <p className="text-gray-600"><span className="font-medium text-gray-900">Nom:</span> {agent.firstName} {agent.lastName}</p>
                                    <p className="text-gray-600"><span className="font-medium text-gray-900">Email:</span> {agent.email}</p>
                                    <button className="w-full mt-4 py-2 px-4 rounded-md text-sm font-medium text-white bg-gray-900 hover:bg-black">
                                        Contacter l'agent
                                    </button>
                                </div>
                            ) : (
                                <p className="mt-4 text-gray-600">Information indisponible.</p>
                            )}
                        </div>
                    </aside>
                </main>
            </div>
        </div>
    );
}